import {expect, Locator, test} from "@playwright/test";
import {formEncodeObject, obtainCSRFToken} from "../utils/AemRequestHelper";

export default class TranslationJobCardComponent {

  private readonly element: Locator;
  private readonly optionsDropdown: Locator;
  private readonly detailsButton: Locator;
  private readonly statusCell: Locator;
  private readonly targetCell: Locator;
  private readonly pagesCell: Locator;
  private readonly assetsCell: Locator;
  private readonly contentFragmentsCell: Locator;
  private csrfToken;

  constructor(element: Locator) {
    this.element = element;
    this.optionsDropdown = element.locator('.cq-projects-tile-actions');
    this.detailsButton = element.locator('footer > a');
    this.statusCell = element.locator('.cell-status-value').first();
    this.targetCell = element.locator('.cell-status-value').nth(1);
    this.pagesCell = element.locator('.cell-status-value').nth(2);
    this.assetsCell = element.locator('.cell-status-value').nth(3);
    this.contentFragmentsCell = element.locator('.cell-status-value').nth(4);
  }

  async requestScope() {
    await this.pickDropdownOption('Request Scope');
  }

  async startJob() {
    await this.pickDropdownOption('Start');
    await this.waitForJobToBeStarted();
  }

  private async pickDropdownOption(option: string) {
    await this.optionsDropdown.click();
    const popoverId = await this.optionsDropdown.getAttribute('id');
    const popover = this.element.page().locator(`[target="#${popoverId}"]`);
    await expect(popover).toBeVisible();
    const button = popover.getByText(option);
    if (await button.isVisible()) {
      await button.click();
    } else {
      test.fail();
    }
  }

  async addTranslationResources(paths: string[]) {
    const page = this.element.page();
    const request = page.request;
    const translationJobPath = await this.element.getAttribute('data-path');

    if (!this.csrfToken) {
      this.csrfToken = await obtainCSRFToken(page);
    }

    const options = {
      data: this.prepareFormData(translationJobPath, paths), // we cannot use form element due to fact that translationpage param is used many times
      headers: {
        'content-type': 'application/x-www-form-urlencoded; charset=UTF-8',
        'sec-fetch-mode': 'cors',
        'referrerPolicy': 'strict-origin-when-cross-origin',
        'csrf-token': this.csrfToken,
        'referer': page.url(),
      }
    }
    const apiResponse = await request.post(`${process.env.AEM_AUTHOR_URL}${translationJobPath}`, options);
    expect(apiResponse.ok()).toBeTruthy();
    expect(apiResponse.status()).toEqual(200);

    await expect.poll(async () => {
      await this.element.page().reload();
      return await this.pagesCell.innerText();
    }, {
      message: 'Checking if resources have been added to translation job card component',
      intervals: [1_000, 5_000, 15_000, 30_000],
      timeout: 60_000,
    }).not.toBe('0');
  }

  async waitForJobToBeStarted() {
    await this.waitForJobStatusChange().not.toBe('Draft');
  }

  async waitForJobToBeStarted() {
    await expect.poll(async () => {
      await this.element.page().reload();
      return await this.statusCell.innerText();
    }, {
      message: 'Checking translation status',
      intervals: [1_000, 5_000, 15_000, 30_000],
      timeout: 300_000,
    }).not.toBe('Draft')
  }

  async waitForJobToFinish() {
    await this.waitForJobStatusChange().toBe('Approved');
  }

  waitForJobStatusChange() {
    return expect.poll(async () => {
      await this.element.page().reload();
      return await this.statusCell.innerText();
    }, {
      message: 'Checking translation status',
      intervals: [1_000, 5_000, 15_000, 30_000],
      timeout: 300_000,
    });
  }

  private prepareFormData(translationJobPath: string, paths: string[]) {
    const formData = {
      ':translationJobPath': translationJobPath,
      '_charset_': 'UTF-8',
      ':operation': 'ADD_TRANSLATION_PAGES',
      type: 'ASSET/SITE',
      createLanguageCopy: true,
      allowChildren: true
    }
    let formEncodedBody = formEncodeObject(formData) + '&'
    formEncodedBody += paths.map(p => 'translationpage=' + encodeURIComponent(p)).join('&');
    return formEncodedBody;
  }
}