/******************************************
 *
 * Copyright (c) 2003-2010 XML-INTL Ltd.
 *
 * All Rights Reserved
 *
 ******************************************/

/* global jQuery */

(function ($, document) {
  "use strict";

  var $form,
      $connectionDialog,
      $connectionDialogHeader,
      $connectionDialogContent,
      $xtmWebserviceURI,
      $xtmClientName,
      $xtmUserId,
      $xtmPassword,
      $selectCustomerId,
      $selectTemplateId,
      $connectButton;

  function
  init() {
    bindDOM();
    bindEvents();
    setButtonState();

    if (canConnect()) {
      populateCustomerList().then(function (data) {
        if (data.selected) {
          populateTemplatesList(data.selected);
        }
      });
    }
  }

  function bindDOM() {
    $form = $('#services-editor-form');
    $connectionDialog = $form.find('#connectionDialog');
    $connectionDialogHeader = $connectionDialog.find('coral-dialog-header');
    $connectionDialogContent = $connectionDialog.find('coral-dialog-content');
    $xtmWebserviceURI = $form.find('#xtmWebserviceURI');
    $xtmClientName = $form.find('#xtmClientname');
    $xtmUserId = $form.find('#xtmUserId');
    $xtmPassword = $form.find('#xtmPassword');
    $selectCustomerId = $form.find('#selectCustomerId');
    $selectTemplateId = $form.find('#selectTemplateId');
    $connectButton = $form.find('#connectButton');

    $(window).adaptTo("foundation-registry").register("foundation.form.submit", {
      selector: "*",
      handler: beforeSubmit
    });
  }

  function bindEvents() {
    $xtmWebserviceURI.on('change', setButtonState);
    $xtmClientName.on('change', setButtonState);
    $xtmUserId.on('change', setButtonState);
    $xtmPassword.on('change', setButtonState);
    $connectButton.on('click', connect);

    $connectionDialog.bind('coral-overlay:close', onDialogClose);
    $selectCustomerId.on('change', function (e) {
      populateTemplatesList(e.target.selectedItem.value);
    });
  }

  function beforeSubmit() {
    var deferred = $.Deferred();

    function onError() {
      handleError('Connection Error', 'Failed to connect to XTM - please verify credentials and proxy settings');
      deferred.reject();
    }

    function onSuccess(data) {
      if (!data.verified) {
        onError(data);
      } else {
        deferred.resolve();
      }
    }

    if (!canConnect()) {
      handleError('Connection Error', 'Failed to connect to XTM - please verify credentials and proxy settings');
      deferred.reject();
      return;
    }

    if(!$selectCustomerId.val() || !$selectTemplateId.val()){
      handleError('Connection Error', 'Customer ID or Template ID are not set - please connect to XTM');
      deferred.reject();
    }

    $.post('/services/xtm', getFormData())
        .done(onSuccess)
        .fail(onError);

    return {
      preResult: deferred
    };
  }

  function canConnect() {
    var data = getFormData();
    var keys = Object.keys(data);
    var length = keys.filter(function (key) {
      return data[key];
    }).length;
    return length === 4;
  }

  function setButtonState() {
    if (canConnect()) {
      $connectButton.removeAttr('disabled');
    } else {
      $connectButton.attr('disabled', 'disabled');
    }
  }

  function connect(e) {
    e.preventDefault();
    if (!canConnect()) {
      return;
    }

    function onError(data) {
      changeIcon('close');
      handleError('Error', data.errorMessage)
    }

    function onSuccess(data) {
      if (data.verified) {
        changeIcon('check');
        populateCustomerList();
      } else {
        onError(data);
      }
    }

    changeIcon('gear', true);
    return $.post('/services/xtm', getFormData())
        .done(onSuccess)
        .fail(onError);
  }

  function populateCustomerList() {
    return populateList($selectCustomerId, getCustomerList);
  }

  function populateTemplatesList(selectedCustomerId) {
    return populateList($selectTemplateId, getTemplatesList,
        selectedCustomerId);
  }

  function populateList($field, promise, promiseArguments) {
    return new Promise(function (resolve, reject) {
      if (!$field[0]) {
        reject();
      }

      var selectedValue = $field.data('selected');
      var field = $field[0];
      if (field.items) {
        field.items.clear();
      }

      promise(promiseArguments).then(function (items) {
        items.forEach(function (item) {
          var selectItem = {
            value: item.id,
            content: {textContent: item.name},
            selected: parseInt(item.id) === selectedValue
          };
          field.items.add(selectItem);
        });

        if (items.length > 0) {
          $field.removeAttr('disabled');
        } else {
          $field.attr('disabled', 'disabled');
        }

        resolve({selected: selectedValue, items: items});
      })
    });
  }

  function getFormData() {
    return {
      endpoint: $xtmWebserviceURI.val(),
      client: $xtmClientName.val(),
      userid: $xtmUserId.val(),
      pwd: $xtmPassword.val()
    };
  }

  function getCustomerList() {
    // TODO remove urlParam once we deal with statefullness of XtmDropdownDataProviderService
    var urlParam = getFormData();
    return $.get('/services/xtm.customerList.json', urlParam)
        .then(function (data, status) {
          if (status === 'success') {
            return (data && data.customers) ? data.customers : [];
          } else {
            handleError('Connection failed', 'Connection failed: HTTP ' + status);
          }
        })
        .fail(handleRequestFailure);
  }

  function getTemplatesList(selectCustomerId) {
    var urlParam = {'custid': selectCustomerId};
    return $.get('/services/xtm.workflowList.json', urlParam)
        .then(function (data, status) {
          if (status === 'success') {
            return (data && data.workflows) ? data.workflows : [];
          } else {
            handleError('Connection failed', 'Connection failed: HTTP ' + status);
          }
        })
        .fail(handleRequestFailure);
  }

  function onDialogClose() {
    changeIcon('publish');
  }

  function changeIcon(icon, spin) {
    $connectButton.attr('icon', icon);
    (spin)
        ? $connectButton.addClass('loading')
        : $connectButton.removeClass('loading');
  }

  function handleRequestFailure(response) {
    handleError('HTTP ' + response.status,
        '<b> Status:</b> ' + response.statusText +
        ' <br/> <b>Reason:</b> ' + response.error().responseText);
  }

  function handleError(title, message) {
    $connectionDialogHeader.text(title);
    $connectionDialogContent.html(message || 'Unknown error');
    $connectionDialog[0].show();
    console.error(title, message);
  }

  $(document).ready(function () {
    var $elem = $('#services-editor-form');
    if ($elem.length) {
      init();
    }
  });

})(jQuery, document);


