/* global jQuery */

(function ($, document) {
  "use strict";

  var ui = $(window).adaptTo('foundation-ui');

  var $form,
      $connectionDialog,
      $connectionDialogHeader,
      $connectionDialogContent,
      $xtmWebserviceURI,
      $xtmClientName,
      $xtmUserId,
      $xtmPassword,
      $selectCustomerId,
      $selectTemplateId,
      $connectButton,
      $projectCreationStrategyDropdown,
      $launchHandlingStrategyDropdown;

  function
  init() {
    bindDOM();
    bindEvents();
    setButtonState();
    updateProjectCreationStrategyDropdown();

    if (canConnect()) {
      populateCustomerList().then(function (data) {
        if (data.selected) {
          populateTemplatesList(data.selected);
        }
      });
    }
  }

  function bindDOM() {
    $form = $('#services-editor-form');
    $connectionDialog = $form.find('#connectionDialog');
    $connectionDialogHeader = $connectionDialog.find('coral-dialog-header');
    $connectionDialogContent = $connectionDialog.find('coral-dialog-content');
    $xtmWebserviceURI = $form.find('#xtmWebserviceURI');
    $xtmClientName = $form.find('#xtmClientname');
    $xtmUserId = $form.find('#xtmUserId');
    $xtmPassword = $form.find('#xtmPassword');
    $selectCustomerId = $form.find('#selectCustomerId');
    $selectTemplateId = $form.find('#selectTemplateId');
    $connectButton = $form.find('#connectButton');
    $projectCreationStrategyDropdown = $form.find('#projectCreationStrategyRaw');
    $launchHandlingStrategyDropdown = $form.find('#launchHandlingStrategyRaw');
  }

  function bindEvents() {
    $xtmWebserviceURI.on('change', setButtonState);
    $xtmClientName.on('change', setButtonState);
    $xtmUserId.on('change', setButtonState);
    $xtmPassword.on('change', setButtonState);
    $connectButton.on('click', connect);

    $connectionDialog.bind('coral-overlay:close', onDialogClose);
    $selectCustomerId.on('change', function (e) {
      populateTemplatesList(e.target.selectedItem.value);
    });
    $projectCreationStrategyDropdown.on('change', updateProjectCreationStrategyDropdown);

    $(window).adaptTo('foundation-registry').register('foundation.form.submit', {
      selector: '.xtm-translation-cloud-configuration-form',
      handler: beforeSubmit
    });
  }

  function beforeSubmit() {
    var deferred = $.Deferred();

    if (!canConnect()) {
      handleError('Connection Error', 'Failed to connect to XTM - please verify credentials and proxy settings');
      deferred.reject();
    }

    if (!$selectCustomerId.val()) {
      handleError('Connection Error', 'Customer ID is not set - please connect to XTM');
      deferred.reject();
    }

    if (!$selectTemplateId.val()) {
      handleError('Connection Error', 'Template ID is not set - please connect to XTM');
      deferred.reject();
    }

    deferred.resolve();

    return {
      preResult: deferred
    };
  }

  function canConnect() {
    var data = getFormData();
    var keys = Object.keys(data);
    var length = keys.filter(function (key) {
      return data[key];
    }).length;
    return length === 4;
  }

  function setButtonState() {
    clearList($selectCustomerId);
    clearList($selectTemplateId);
    if (canConnect()) {
      $connectButton.removeAttr('disabled');
    } else {
      $connectButton.attr('disabled', 'disabled');
    }
  }

  function connect(e) {
    e.preventDefault();
    if (!canConnect()) {
      return;
    }

    function onError(data) {
      changeIcon('close');
      handleError('Error', data.errorMessage)
    }

    function onSuccess() {
      changeIcon('check');
    }

    changeIcon('gear', true);
    return populateCustomerList()
        .then(onSuccess)
        .fail(onError);
  }

  function updateProjectCreationStrategyDropdown() {
    var currentValue = $projectCreationStrategyDropdown[0].value
    if (currentValue === 'NEW_PROJECT') {
      $launchHandlingStrategyDropdown[0].value = 'SEPARATE';
      $launchHandlingStrategyDropdown.attr('disabled', 'disabled');
    } else if (currentValue === 'PROJECT_UPDATE') {
      $launchHandlingStrategyDropdown.removeAttr('disabled');
    }
  }

  function populateCustomerList() {
    return populateList($selectCustomerId, getCustomerList);
  }

  function populateTemplatesList(selectedCustomerId) {
    return populateList($selectTemplateId, getTemplatesList,
        selectedCustomerId);
  }

  function clearList($field) {
    var field = $field[0];
    if (field.items) {
      field.items.clear();
    }
  }

  function populateList($field, promise, promiseArguments) {
    return new Promise(function (resolve, reject) {
      if (!$field[0]) {
        reject();
      }

      var selectedValue = $field.data('selected');
      var field = $field[0];
      if (field.items) {
        field.items.clear();
      }

      promise(promiseArguments).then(function (items) {
        items.forEach(function (item) {
          var selectItem = {
            value: item.id,
            content: {textContent: item.name},
            selected: parseInt(item.id) === selectedValue
          };
          field.items.add(selectItem);
        });

        if (items.length > 0) {
          $field.removeAttr('disabled');
        } else {
          $field.attr('disabled', 'disabled');
        }

        resolve({selected: selectedValue, items: items});
      })
    });
  }

  function getFormData() {
    return {
      xtmWebserviceURI: $xtmWebserviceURI.val(),
      xtmClientname: $xtmClientName.val(),
      xtmUserId: $xtmUserId.val(),
      xtmPassword: $xtmPassword.val()
    };
  }

  function getCustomerList() {
    var data = getFormData();
    return $.post('/services/xtm/customerList.json', data)
        .then(function (data, status) {
          if (status === 'success') {
            return (data) ? data : [];
          } else {
            handleError('Connection Error', 'Connection failed: HTTP ' + status);
          }
        })
        .fail(handleRequestFailure);
  }

  function getTemplatesList(selectedCustomerId) {
    var data = getFormData();
    data.customerId = selectedCustomerId;
    return $.post('/services/xtm/projectTemplateList.json', data)
        .then(function (data, status) {
          if (status === 'success') {
            return (data) ? data : [];
          } else {
            handleError('Connection failed', 'Connection failed: HTTP ' + status);
          }
        })
        .fail(handleRequestFailure);
  }

  function onDialogClose() {
    changeIcon('publish');
  }

  function changeIcon(icon, spin) {
    $connectButton.attr('icon', icon);
    (spin)
        ? $connectButton.addClass('loading')
        : $connectButton.removeClass('loading');
  }

  function handleRequestFailure(response) {
    handleError('HTTP ' + response.status,
        '<b> Status:</b> ' + response.statusText +
        ' <br/> <b>Reason:</b> ' + response.error().responseText);
  }

  function handleError(title, message) {
    $connectionDialogHeader.text(title);
    $connectionDialogContent.html(message || 'Unknown error');
    $connectionDialog[0].show();
    console.error(title, message);
  }

  $(document).ready(function () {
    var $elem = $('#services-editor-form');
    if ($elem.length) {
      init();
    }
  });

})(jQuery, document);


