import BasePage from "./BasePage";
import {expect, Locator} from "@playwright/test";
import ProjectCreationWizardForm from "./ProjectCreationWizardForm";
import CoralSelectComponent from "./CoralSelectComponent";
import CoralPathfieldComponent from "./CoralPathfieldComponent";
import ProjectPage from "./ProjectPage";

const WIZARD_PAGE_PATH = '/mnt/overlay/cq/core/content/projects/wizard/newproject.html/content/projects';
const TRANSLATION_PROJECT_TEMPLATE = '/libs/cq/core/content/projects/templates/translation-project';
export default class ProjectCreationWizardPage extends BasePage {

  private readonly projectCard: Locator;
  private readonly nextButton: Locator;
  private readonly titleField: Locator;
  private readonly descriptionField: Locator;
  private readonly basicTab: Locator;
  private readonly automaticallyApproveTranslationField: Locator;
  private readonly advancedTab: Locator;
  private readonly nameField: Locator;
  private readonly sourceLanguageField: CoralSelectComponent;
  private readonly targetLanguagesField: CoralSelectComponent;
  private readonly cloudConfigurationField: CoralPathfieldComponent;
  private readonly translationMethodField: CoralSelectComponent;
  private readonly translationProviderField: CoralSelectComponent;
  private readonly automaticallyPromoteTranslationLaunchesField: Locator;
  private readonly deleteLaunchAfterPromotionField: Locator;
  private readonly createButton: Locator;
  private readonly modalSuccess: Locator;

  constructor(page) {
    super(page, WIZARD_PAGE_PATH, '.cq-projects-admin-details');
    this.projectCard = this.page.locator(`[data-granite-collection-item-id='${TRANSLATION_PROJECT_TEMPLATE}']`);
    this.nextButton = this.page.locator(`button[data-foundation-wizard-control-action='next'][type='button']`);
    this.createButton = this.page.getByRole('button', {name: 'Create'});
    this.titleField = this.page.getByLabel('Title');
    this.descriptionField = this.page.getByLabel('Description', {exact: true});
    this.basicTab = this.page.getByText('Basic');
    this.advancedTab = this.page.getByText('Advanced');
    this.nameField = this.page.getByLabel('Name *');
    this.sourceLanguageField = new CoralSelectComponent(this.page.locator('coral-select[name="sourceLanguage"]'));
    this.targetLanguagesField = new CoralSelectComponent(this.page.locator('coral-select[name="processProperty"]'));
    this.cloudConfigurationField = new CoralPathfieldComponent(this.page.locator('foundation-autocomplete[name="./cq:conf"]'));
    this.translationMethodField = new CoralSelectComponent(this.page.locator('coral-select[name="translationMethod"]'));
    this.translationProviderField = new CoralSelectComponent(this.page.locator('coral-select[name="translationProvider"]'));
    this.automaticallyPromoteTranslationLaunchesField = this.page.getByLabel('Automatically Promote Translation Launches');
    this.deleteLaunchAfterPromotionField = this.page.getByLabel('Delete Launch After Promotion');
    this.automaticallyApproveTranslationField = this.page.getByLabel('Automatically Approve Translations');
    this.modalSuccess = this.page.getByText('Success The project was createdDoneOpen');
  }

  async pickTranslationProject() {
    await expect(this.nextButton).toBeDisabled();
    await this.projectCard.click();
  }

  async clickNext() {
    await expect(this.nextButton).toBeEnabled();
    await this.nextButton.click();
    await expect(this.page.getByRole('alert')).toBeHidden();
  }

  async clickCreate(): Promise<ProjectPage> {
    await expect(this.createButton).toBeEnabled();
    await this.createButton.click();
    await expect(this.modalSuccess).toBeVisible();
    await this.modalSuccess.getByRole('button', {name: 'Open'}).click();
    await this.waitForPageLoad();
    return new ProjectPage(this.page);
  }

  async fillForm(form: ProjectCreationWizardForm) {
    // fill basic tab
    await expect(this.titleField).toBeVisible();
    await this.titleField.fill(form.title);
    await this.descriptionField.fill(form.description);
    await this.advancedTab.click();

    // fill advanced tab
    await expect(this.nameField).toBeVisible();
    await this.nameField.fill(form.name);
    await this.sourceLanguageField.select([form.sourceLanguage]);
    await this.targetLanguagesField.select(form.targetLanguages);
    await this.cloudConfigurationField.fill(form.cloudConfiguration);
    await this.translationMethodField.select([form.translationMethod]);
    await this.translationProviderField.select([form.translationProvider]);
    await this.automaticallyPromoteTranslationLaunchesField.setChecked(form.automaticallyApproveTranslation);
    await this.deleteLaunchAfterPromotionField.setChecked(form.deleteLaunchAfterPromotion);
    await this.automaticallyApproveTranslationField.setChecked(form.automaticallyApproveTranslation);
  };
}