import {expect, Locator, Page} from "@playwright/test";
import XtmBasePage from "./XtmBasePage";

export default class XtmProjectsPage extends XtmBasePage {
  private readonly table: Locator;
  private readonly actionsButton: Locator;
  private readonly tableItems: Locator;

  constructor(page: Page) {
    super(page, `/project-manager-gui/list-projects-page.action`, '#projects_table');
    this.table = this.page.locator('#projects_table');
    this.actionsButton = this.page.getByRole('link', {name: 'Actions'});
    this.tableItems = this.table.locator('tbody > tr');
  }

  async finishProject(name: string) {
    const item = await this.waitForProjectToArrive(name);
    await this.waitForProjectToStart(item);
    await this.dropdownClickToFinishProject(item);
  }

  private async waitForProjectToArrive(name: string) {
    let item: Locator = null;
    await expect.poll(async () => {
      await this.page.reload();
      item = this.tableItems.filter({hasText: name}).first();
      return item;
    }, {
      message: `Waiting for XTM project ${name} to arrive`,
      intervals: [1_000, 5_000, 15_000, 30_000],
      timeout: 60_000,
    }).not.toBeNull();
    await expect(item).not.toBeNull();
    return item;
  }

  private async waitForProjectToStart(projectRow: Locator) {
    await expect.poll(async () => {
      await this.page.reload();
      return await projectRow.locator('.projects_table_MAX_STATUS').innerText();
    }, {
      message: 'Checking XTM project MAX translation status',
      intervals: [1_000, 5_000, 15_000, 30_000],
      timeout: 60_000,
    }).not.toBe('Not started');
  }

  private async dropdownClickToFinishProject(item: Locator) {
    const itemId = await item.getAttribute('id');
    const itemMenuButton = item.locator('.projects_table_MENU');
    await itemMenuButton.click();
    const itemMenu = await this.page.locator(`#context-menu-${itemId}`);
    await expect(item).toBeVisible();
    const itemMenuActionsItem = itemMenu.locator('li', {hasText: 'Actions'});
    await itemMenuActionsItem.click();
    const subMenu = await itemMenuActionsItem.locator('ul.submenu');
    await expect(subMenu).toBeVisible();
    await subMenu.getByText('Finish project').click();
    const confirmationModal = await this.page.getByLabel('Confirm finishing the project');
    await expect(confirmationModal).toBeVisible();
    await confirmationModal.getByRole('button', {name: 'OK'}).click();
  }
}