import ProjectCreationWizardPage from "../page-objects/ProjectCreationWizardPage";
import ProjectCreationWizardForm from "../page-objects/ProjectCreationWizardForm";
import XtmProjectsPage from "../page-objects/XtmProjectsPage";
import {expect, Page, test} from "@playwright/test";
import {AUTH_FILE_XTM} from "./consts";
import conf from '../resources/translate-journey.data';
import ProjectPage from "../page-objects/ProjectPage";

test.describe('Translation journey - create project', () => {

  test('single language [fr]', async ({page, browser}) => {
    const payload = conf.fr;
    const form = new ProjectCreationWizardForm();
    form.targetLanguages.push('French');
    form.description = 'single language [fr]';

    const projectPage = await executeProjectWizardSteps(page, form);
    await executeProjectPageStartSteps(projectPage, payload);
    await executeXtmProjectSteps(browser, form);
    await executeProjectPageFinishSteps(projectPage);
    await assertResults(page, payload);
  });

  test('multiple languages [es,de,it]', async ({page, browser}) => {
    const payloadItems = [conf.es, conf.de, conf.it];
    const form = new ProjectCreationWizardForm();
    form.targetLanguages.push('Spanish', 'German', 'Italian');
    form.description = 'multiple languages [es,de,it]';

    const projectPage = await executeProjectWizardSteps(page, form);

    for (const payload of payloadItems) {
      await executeProjectPageStartSteps(projectPage, payload);
    }

    await executeXtmProjectSteps(browser, form);

    for (const payload of payloadItems) {
      await executeProjectPageFinishSteps(projectPage);
      await assertResults(page, payload);
    }
  });

  async function executeProjectWizardSteps(page: Page, form: ProjectCreationWizardForm): Promise<ProjectPage> {
    const projectWizardPage = new ProjectCreationWizardPage(page);
    await projectWizardPage.open();
    await projectWizardPage.pickTranslationProject();
    await projectWizardPage.clickNext();
    await projectWizardPage.fillForm(form);
    return await projectWizardPage.clickCreate();
  }

  async function executeProjectPageStartSteps(projectPage: ProjectPage, payload) {
    await projectPage.addTranslationResources(payload.map(item => item.source));
    await projectPage.startAllJobs();
  }

  async function executeXtmProjectSteps(browser, form) {
    const xtmContext = await browser.newContext({storageState: AUTH_FILE_XTM});
    const xtmPage = await xtmContext.newPage();
    const xtmProjectsPage = new XtmProjectsPage(xtmPage);
    await xtmProjectsPage.open();
    await xtmProjectsPage.finishProject(form.title);
    await xtmContext.close();
  }

  async function executeProjectPageFinishSteps(projectPage) {
    await projectPage.open();
    await projectPage.waitForAllJobsToFinish();
  }

  async function assertResults(page, payload) {
    for (const item of payload) {
      const url = `${process.env.AEM_AUTHOR_URL}${item.target.path}.html?wcmmode=disabled`;
      await page.goto(url);
      expect(await page.title()).toEqual(item.target.title);
      for (const text of item.target.content) {
        expect(await page.content()).toEqual(expect.stringContaining(text));
      }
    }
  }
});